/*
 * Copyright (c) 2024-2026 Actinius
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <math.h>
#include <zephyr/kernel.h>
#include <zephyr/device.h>
#include <zephyr/drivers/sensor.h>

#ifdef CONFIG_ACTINIUS_ACCEL_USE_ANSI
/* ANSI escape codes for terminal control */
#define ANSI_CURSOR_HOME "\033[1;1H"
#define ANSI_CLEAR_SCREEN "\033[2J"
#endif

#define ACCEL_NODE DT_ALIAS(accel0)

typedef enum {
	ORIENTATION_UNKNOWN,
	ORIENTATION_FACE_UP,
	ORIENTATION_FACE_DOWN,
	ORIENTATION_LANDSCAPE,
	ORIENTATION_LANDSCAPE_INVERTED,
	ORIENTATION_PORTRAIT_LEFT,
	ORIENTATION_PORTRAIT_RIGHT,
} orientation_t;

static const struct device *accel;
static orientation_t current_orientation = ORIENTATION_UNKNOWN;
static struct sensor_trigger trig;

static const char *orientation_to_string(orientation_t orientation)
{
	switch (orientation) {
	case ORIENTATION_FACE_UP:
		return "Face Up";
	case ORIENTATION_FACE_DOWN:
		return "Face Down";
	case ORIENTATION_LANDSCAPE:
		return "Landscape";
	case ORIENTATION_LANDSCAPE_INVERTED:
		return "Landscape (Inverted)";
	case ORIENTATION_PORTRAIT_LEFT:
		return "Portrait Left";
	case ORIENTATION_PORTRAIT_RIGHT:
		return "Portrait Right";
	default:
		return "Unknown";
	}
}

static orientation_t calculate_orientation(double x, double y, double z)
{
	/* Threshold for detecting dominant axis (about 0.7g) */
	const double threshold = 7.0;

	if (z > threshold) {
		return ORIENTATION_FACE_UP;
	} else if (z < -threshold) {
		return ORIENTATION_FACE_DOWN;
	} else if (y > threshold) {
		return ORIENTATION_LANDSCAPE;
	} else if (y < -threshold) {
		return ORIENTATION_LANDSCAPE_INVERTED;
	} else if (x > threshold) {
		return ORIENTATION_PORTRAIT_LEFT;
	} else if (x < -threshold) {
		return ORIENTATION_PORTRAIT_RIGHT;
	}

	return ORIENTATION_UNKNOWN;
}

static void accel_trigger_handler(const struct device *dev,
				  const struct sensor_trigger *trig)
{
	struct sensor_value val_x, val_y, val_z;
	double x, y, z;
	orientation_t new_orientation;

	if (sensor_sample_fetch(dev) < 0) {
		printk("Failed to fetch sample\n");
		return;
	}

	sensor_channel_get(dev, SENSOR_CHAN_ACCEL_X, &val_x);
	sensor_channel_get(dev, SENSOR_CHAN_ACCEL_Y, &val_y);
	sensor_channel_get(dev, SENSOR_CHAN_ACCEL_Z, &val_z);

	x = sensor_value_to_double(&val_x);
	y = sensor_value_to_double(&val_y);
	z = sensor_value_to_double(&val_z);

	new_orientation = calculate_orientation(x, y, z);

	if (new_orientation != ORIENTATION_UNKNOWN &&
	    new_orientation != current_orientation) {
		current_orientation = new_orientation;
#ifdef CONFIG_ACTINIUS_ACCEL_USE_ANSI
		printk(ANSI_CURSOR_HOME);
		printk(ANSI_CLEAR_SCREEN);
		printk("Accelerometer Interrupt Sample\n");
		printk("-------------------------------------------------------------------------------\n");
#endif
		printk("Orientation: %s\n", orientation_to_string(current_orientation));
		printf("    (X: %.2f, Y: %.2f, Z: %.2f m/s^2)\n", x, y, z);
#ifdef CONFIG_ACTINIUS_ACCEL_USE_ANSI
		printk("-------------------------------------------------------------------------------\n");
#endif
	}
}

int main(void)
{
	int err;

#ifdef CONFIG_ACTINIUS_ACCEL_USE_ANSI
	printk(ANSI_CURSOR_HOME);
	printk(ANSI_CLEAR_SCREEN);
#endif
	printk("Accelerometer Interrupt Sample\n");
	printk("Detecting orientation changes via interrupt...\n\n");

	accel = DEVICE_DT_GET(ACCEL_NODE);
	if (!device_is_ready(accel)) {
		printk("Error: accelerometer device not ready\n");
		return -1;
	}

	/* Configure slope threshold for motion detection */
	struct sensor_value slope_th = { .val1 = 2, .val2 = 0 };
	struct sensor_value slope_dur = { .val1 = 1, .val2 = 0 };

	sensor_attr_set(accel, SENSOR_CHAN_ACCEL_XYZ,
			SENSOR_ATTR_SLOPE_TH, &slope_th);
	sensor_attr_set(accel, SENSOR_CHAN_ACCEL_XYZ,
			SENSOR_ATTR_SLOPE_DUR, &slope_dur);

	/* Set up any-motion trigger */
	trig.type = SENSOR_TRIG_DELTA;
	trig.chan = SENSOR_CHAN_ACCEL_XYZ;

	printk("Accelerometer trigger configured.\n");
	printk("Rotate the board to detect orientation changes.\n\n");

	err = sensor_trigger_set(accel, &trig, accel_trigger_handler);
	if (err) {
		printk("Failed to set trigger: %d\n", err);
		return -1;
	}

	/* Main loop just sleeps - all work is done in the trigger handler */
	while (1) {
		k_sleep(K_FOREVER);
	}

	return 0;
}
