#
# Copyright (c) 2024-2026 Actinius
#
# SPDX-License-Identifier: BSD-3-Clause
#

# This macro must be called AFTER find_package(Zephyr) since BOARD/BOARD_QUALIFIERS
# are only available after Zephyr's build system processes them.
macro(actinius_check_board)
  if(NOT DEFINED ACTINIUS_SUPPORTED_BOARDS)
    message(FATAL_ERROR "ACTINIUS_SUPPORTED_BOARDS not set")
  endif()

  set(_board_match FALSE)
  # In HWMv2: BOARD = board name, BOARD_QUALIFIERS = /soc/variant
  # Full board target is ${BOARD}${BOARD_QUALIFIERS}
  set(_board_target "${BOARD}${BOARD_QUALIFIERS}")

  foreach(b ${ACTINIUS_SUPPORTED_BOARDS})
    # Check if the supported board entry specifies a revision (contains @)
    string(FIND "${b}" "@" _at_pos)

    if(_at_pos GREATER -1)
      # Entry has revision requirement: extract board and revision
      string(SUBSTRING "${b}" 0 ${_at_pos} _req_board)
      math(EXPR _rev_start "${_at_pos} + 1")
      string(SUBSTRING "${b}" ${_rev_start} -1 _req_revision)

      # Match both board target and revision
      if("${_board_target}" STREQUAL "${_req_board}" AND "${BOARD_REVISION}" STREQUAL "${_req_revision}")
        set(_board_match TRUE)
      endif()
    else()
      # No revision specified: match board target only (any revision OK)
      if("${_board_target}" STREQUAL "${b}")
        set(_board_match TRUE)
      endif()
    endif()
  endforeach()

  if(NOT _board_match)
    set(_board_display "${_board_target}")
    if(DEFINED BOARD_REVISION AND NOT "${BOARD_REVISION}" STREQUAL "")
      set(_board_display "${_board_target}@${BOARD_REVISION}")
    endif()
    message(FATAL_ERROR
      "\n========================================\n"
      " Unsupported board: ${_board_display}\n"
      " This sample supports: ${ACTINIUS_SUPPORTED_BOARDS}\n"
      "========================================\n")
  endif()
endmacro()
