/*
 * Copyright (c) 2019-2026 Actinius
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <string.h>
#include <zephyr/kernel.h>
#include <zephyr/net/socket.h>
#include <zephyr/net/tls_credentials.h>
#include <net/rest_client.h>
#include <modem/nrf_modem_lib.h>
#include <modem/lte_lc.h>
#include <modem/modem_key_mgmt.h>
#include <zephyr/data/json.h>

static const char *iss_position_api_host = "api.open-notify.org";
static const char *iss_position_api_endpoint = "/iss-now.json";

static const char *reverse_geolocation_api_host = "nominatim.openstreetmap.org";

#define TLS_SEC_TAG 301222

struct iss_position {
	char *latitude;
	char *longitude;
};

struct iss_position_api_response {
	char *message;
	struct iss_position iss_position;
	uint32_t timestamp;
};

static const struct json_obj_descr iss_position_descriptor[] = {
	JSON_OBJ_DESCR_PRIM(struct iss_position, latitude, JSON_TOK_STRING),
	JSON_OBJ_DESCR_PRIM(struct iss_position, longitude, JSON_TOK_STRING),
};

static const struct json_obj_descr iss_position_api_response_descriptor[] = {
	JSON_OBJ_DESCR_PRIM(struct iss_position_api_response, message, JSON_TOK_STRING),
	JSON_OBJ_DESCR_OBJECT(struct iss_position_api_response, iss_position,
			      iss_position_descriptor),
	JSON_OBJ_DESCR_PRIM(struct iss_position_api_response, timestamp, JSON_TOK_NUMBER),
};

struct reverse_geo_response {
	uint32_t place_id;
	char *display_name;
	char *error;
};

static const struct json_obj_descr reverse_geo_response_descriptor[] = {
	JSON_OBJ_DESCR_PRIM(struct reverse_geo_response, place_id, JSON_TOK_NUMBER),
	JSON_OBJ_DESCR_PRIM(struct reverse_geo_response, display_name, JSON_TOK_STRING),
	JSON_OBJ_DESCR_PRIM(struct reverse_geo_response, error, JSON_TOK_STRING),
};

static char resp_buf[1024];

static const char cert[] = {
	#include "../cert/isrg-root-x1.pem"
};

static int cert_provision(void)
{
	int err;
	bool exists;

	err = modem_key_mgmt_exists(TLS_SEC_TAG, MODEM_KEY_MGMT_CRED_TYPE_CA_CHAIN, &exists);
	if (err) {
		printk("Failed to check for certificates, err %d\n", err);
		return err;
	}

	if (exists) {
		int mismatch = modem_key_mgmt_cmp(TLS_SEC_TAG,
						  MODEM_KEY_MGMT_CRED_TYPE_CA_CHAIN,
						  cert, strlen(cert));
		if (!mismatch) {
			return 0;
		}

		err = modem_key_mgmt_delete(TLS_SEC_TAG, MODEM_KEY_MGMT_CRED_TYPE_CA_CHAIN);
		if (err) {
			printk("Failed to delete existing certificate, err %d\n", err);
		}
	}

	err = modem_key_mgmt_write(TLS_SEC_TAG, MODEM_KEY_MGMT_CRED_TYPE_CA_CHAIN,
				   cert, sizeof(cert) - 1);
	if (err) {
		printk("Failed to provision certificate, err %d\n", err);
		return err;
	}

	return 0;
}

static bool get_iss_position(struct iss_position_api_response *result)
{
	const char *headers[] = {
		"Content-Type: application/json\r\n",
		NULL
	};

	struct rest_client_req_context req = {
		.header_fields = headers,
		.http_method = HTTP_GET,
		.host = iss_position_api_host,
		.url = iss_position_api_endpoint,
		.port = 80,
		.body = NULL,
		.sec_tag = SEC_TAG_TLS_INVALID,
		.tls_peer_verify = TLS_PEER_VERIFY_NONE,
		.resp_buff = resp_buf,
		.resp_buff_len = sizeof(resp_buf),
		.connect_socket = REST_CLIENT_SCKT_CONNECT,
		.keep_alive = false,
		.timeout_ms = 30 * MSEC_PER_SEC,
	};

	struct rest_client_resp_context resp;
	int err;

	err = rest_client_request(&req, &resp);
	if (err) {
		printk("Failed to get ISS position, err %d\n", err);
		return false;
	}

	if (resp.http_status_code != 200) {
		printk("ISS API error, status: %d\n", resp.http_status_code);
		return false;
	}

	err = json_obj_parse(resp.response, resp.response_len,
			     iss_position_api_response_descriptor,
			     ARRAY_SIZE(iss_position_api_response_descriptor), result);
	if (err < 0) {
		printk("ISS JSON parse error, err %d\n", err);
		return false;
	}

	return true;
}

static bool get_reverse_geolocation(struct reverse_geo_response *result,
				    const char *lat, const char *lon)
{
	const char *headers[] = {
		"Content-Type: application/json\r\n",
		"User-Agent: Actinius-IoT-Board\r\n",
		"Accept-Language: en\r\n",
		NULL
	};

	char url[128];

	snprintf(url, sizeof(url),
		 "/reverse?format=json&lat=%s&lon=%s&addressdetails=0&zoom=5", lat, lon);

	struct rest_client_req_context req = {
		.header_fields = headers,
		.http_method = HTTP_GET,
		.host = reverse_geolocation_api_host,
		.url = url,
		.port = 443,
		.body = NULL,
		.sec_tag = TLS_SEC_TAG,
		.tls_peer_verify = TLS_PEER_VERIFY_REQUIRED,
		.resp_buff = resp_buf,
		.resp_buff_len = sizeof(resp_buf),
		.connect_socket = REST_CLIENT_SCKT_CONNECT,
		.keep_alive = false,
		.timeout_ms = 15 * MSEC_PER_SEC,
	};

	struct rest_client_resp_context resp;
	int err;

	err = rest_client_request(&req, &resp);
	if (err) {
		printk("Failed to get reverse geolocation, err %d\n", err);
		return false;
	}

	if (resp.http_status_code != 200) {
		printk("Geolocation API error, status: %d\n", resp.http_status_code);
		return false;
	}

	err = json_obj_parse(resp.response, resp.response_len,
			     reverse_geo_response_descriptor,
			     ARRAY_SIZE(reverse_geo_response_descriptor), result);
	if (err < 0) {
		printk("Geolocation JSON parse error, err %d\n", err);
		return false;
	}

	if (result->error && result->error[0] != '\0') {
		printk("Geolocation API error: %s\n", result->error);
		return false;
	}

	return true;
}

int main(void)
{
	int err;

	printk("ISS Position Sample\n");

	err = nrf_modem_lib_init();
	if (err) {
		printk("Failed to initialize modem library, err %d\n", err);
		return -1;
	}

	err = cert_provision();
	if (err) {
		printk("Failed to provision certificate, err %d\n", err);
		return -1;
	}

	printk("Waiting for network... ");
	err = lte_lc_connect();
	if (err) {
		printk("Failed to connect to the LTE network, err %d\n", err);
		return -1;
	}
	printk("OK\n");

	struct iss_position_api_response iss_result;
	struct reverse_geo_response geo_result;
	char lat[32];
	char lon[32];

	while (true) {
		memset(&iss_result, 0, sizeof(iss_result));
		if (get_iss_position(&iss_result)) {
			strncpy(lat, iss_result.iss_position.latitude, sizeof(lat) - 1);
			strncpy(lon, iss_result.iss_position.longitude, sizeof(lon) - 1);

			printk("ISS Position: %s, %s (timestamp: %d)\n",
			       lat, lon, iss_result.timestamp);

			memset(&geo_result, 0, sizeof(geo_result));
			if (get_reverse_geolocation(&geo_result, lat, lon)) {
				printk("That is over: %s\n", geo_result.display_name);
			}
		}

		k_sleep(K_SECONDS(5));
	}

	return 0;
}
