/*
 * Copyright (c) 2019-2026 Actinius
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <zephyr/drivers/gpio.h>
#include <zephyr/kernel.h>
#include <zephyr/drivers/led_strip.h>
#include <zephyr/device.h>
#include <zephyr/drivers/spi.h>
#include <zephyr/sys/util.h>
#include <string.h>

#define GPIO_NODE			DT_NODELABEL(gpio0)
#define BUTTON_NODE			DT_ALIAS(sw0)

#define STRIP_NODE			DT_ALIAS(led_strip)
#define STRIP_NUM_PIXELS	DT_PROP(DT_ALIAS(led_strip), chain_length)

#define RGB(_r, _g, _b) { .r = (_r), .g = (_g), .b = (_b) }

static const struct led_rgb colors[] = {
	RGB(0x0f, 0x00, 0x00), /* red */
	RGB(0x00, 0x0f, 0x00), /* green */
	RGB(0x00, 0x00, 0x0f), /* blue */
	RGB(0x0f, 0x00, 0x0f), /* magenta */
	RGB(0x00, 0x0f, 0x0f), /* cyan */
	RGB(0x0f, 0x0f, 0x00), /* yellow */
};

struct led_rgb pixels[STRIP_NUM_PIXELS];

static const struct device *gpio_dev;
static struct gpio_callback gpio_cb;

static const struct gpio_dt_spec button = GPIO_DT_SPEC_GET(BUTTON_NODE, gpios);
static const struct device *const strip = DEVICE_DT_GET(STRIP_NODE);

static bool button_pressed = false;

void button_pressed_callback(const struct device *gpiob, struct gpio_callback *cb,
			     gpio_port_pins_t pins)
{
	button_pressed = true;
}

bool init_button(void)
{
	int ret = gpio_pin_configure_dt(&button, GPIO_INPUT);

	if (ret != 0) {
		printk("Error %d: failed to configure %s pin %d\n", ret,
		       button.port->name, button.pin);
		return false;
	}

	ret = gpio_pin_interrupt_configure_dt(&button, GPIO_INT_EDGE_TO_ACTIVE);
	if (ret != 0) {
		printk("Error %d: failed to configure interrupt on %s pin %d\n", ret,
		       button.port->name, button.pin);
		return false;
	}

	gpio_init_callback(&gpio_cb, button_pressed_callback, BIT(button.pin));
	gpio_add_callback(button.port, &gpio_cb);

	return true;
}

int main(void)
{
	size_t color = 0;
	int rc;

	gpio_dev = DEVICE_DT_GET(GPIO_NODE);

	if (!device_is_ready(gpio_dev)) {
		printk("Error: GPIO device not ready\n");
		return -1;
	}

	if (!init_button()) {
		return -1;
	}

	if (!device_is_ready(strip)) {
		printk("LED strip device %s is not ready\n", strip->name);
		return -1;
	}

	/* Make sure LED is off on startup */
	memset(&pixels, 0x00, sizeof(pixels));
	rc = led_strip_update_rgb(strip, pixels, STRIP_NUM_PIXELS);

	while (1) {
		if (button_pressed) {
			memset(&pixels, 0x00, sizeof(pixels));
			memcpy(&pixels, &colors[color], sizeof(pixels));
			rc = led_strip_update_rgb(strip, pixels, STRIP_NUM_PIXELS);

			if (rc) {
				printk("Couldn't update strip: %d\n", rc);
			}

			color++;
			if (color >= ARRAY_SIZE(colors)) {
				color = 0;
			}
			button_pressed = false;
		}

		k_sleep(K_MSEC(100));
	}

	return 0;
}
